/*
File:bl0937.h
Memo:读取计量芯片的电能数据，尝试兼容不同的计量芯片
Date:2023/6/2
Coder:oldfox126@foxmail.com
目的：为新国标五孔计量插座10A(https://oshwhub.com/oldfox126/xin-guo-biao-wu-kong-ji-liang-cha-zuo-10a) 配套的软件工程，本插座使用新国标五孔外壳，相比老款外壳更容易布线。在保留老款特性(磁保持继电器、电能计量、OTA升级固件)的同时，也解决了最后的短板(电流计数不稳定、外壳无法完美贴合等)。是计量插座的完全体。
开源协议: CC-BY-NC-SA 3.0
*/

#include "private/bl0942.h"

uint8_t SerialTemps[22];

void InitReadChip()
{
    //delay(100);
    Serial.begin(4800);
    return;
}

// 每秒钟读一次，依次读(电压或者电流，和功率)。
char cnt = -1;
unsigned char chipIsRead = 0;   //读芯片锁，只有一次读全部完成，才能进行下一次读操作
void readChip()
{
    if (chipIsRead > 0)
    {
        return;
    }

    chipIsRead = 1;
    sendCommandForRead();
    bool flag = readData();
    if (not flag)
    {
        chipIsRead = 0;
        return;
    }

    cnt++;
    float kwh = getEnergy();
    if (0 >= cnt) // first read
    {
        KWh0 = sconfig.KWh - kwh;
        if (0 > KWh0)
        {
            KWh0 = 0;
        }
        sconfig.KWh = KWh0 + kwh;
        needSaveStatus = 1;
        chipIsRead = 0;
        return;
    }

    sconfig.KWh = KWh0 + kwh;
    if (cnt >= 60)
    {
        cnt = 0;
        needSaveStatus = 1;
    }

    chipIsRead = 0;
    return;
}

void sendCommandForRead()
{
    Serial.write((byte)0x58);
    Serial.write((byte)0xAA);
    return;
}

bool readData()
{
    int pre = Serial.read();
    if (-1 == pre)
    {
        return false;
    }

    unsigned char count = 0;
    SerialTemps[0] = readReg();
    while (SerialTemps[0] != 0x55)
    {
        ESP.wdtFeed();
        SerialTemps[0] = readReg();

        count++;
        if (count > 22)
        {
            return false;
        }
    }

    for (byte i = 1; i < 22; i++)
    {
        SerialTemps[i] = readReg();
    }

    return true;
}

String getDebugInfo()
{
    char buffer[300];
    sprintf(buffer, "KWh0=%.4f,KWh=%.4f,Voltage=%.2f,Current=%.2f,Power[%d|%d|%d]=%.2f,Energy=%.4f,FREQ=%.0f,STATUS=%d,rate[Voltage/Current/Power]=%.2f/%.2f/%.2f", KWh0, sconfig.KWh, getVoltage(), getCurrent(), SerialTemps[12], SerialTemps[11], SerialTemps[10], getActivePower(), getEnergy(), getFREQ(), getSTATUS(), mconfig.vrate, mconfig.crate, mconfig.prate);
    return String(buffer);
}

float getCurrent()
{
    uint32_t parm = 0;
    parm = ((uint32_t)SerialTemps[3] << 16) + ((uint32_t)SerialTemps[2] << 8) + SerialTemps[1];
    float current = (float)parm * V_REF * mconfig.crate * 1000 / (305978 * RL_CURRENT); // mA

    return current;
}

float getVoltage()
{
    uint32_t parm = 0;
    parm = ((uint32_t)SerialTemps[6] << 16) + ((uint32_t)SerialTemps[5] << 8) + SerialTemps[4];
    float voltage = (float)parm * V_REF * (R2_VOLTAGE + R1_VOLTAGE) * mconfig.vrate / (73989 * R1_VOLTAGE * 1000);

    return voltage;
}

float getFastCurrent()
{
    uint32_t parm = 0;
    parm = ((uint32_t)SerialTemps[9] << 16) + ((uint32_t)SerialTemps[8] << 8) + SerialTemps[7];
    float fcurrent = (float)parm * V_REF * mconfig.crate * 1000 / (305978 * RL_CURRENT); // mA

    return fcurrent;
}

float getActivePower()
{
    uint32_t parm = 0;
    parm = ((uint32_t)SerialTemps[12] << 16) + ((uint32_t)SerialTemps[11] << 8) + SerialTemps[10];
    byte flag = bitRead(SerialTemps[12], 7);
    if (1 == flag)
    {
        parm = 0xFFFFFF - parm + 1; // 取补码
    }
    float power = (float)parm * mconfig.prate * V_REF * V_REF * (R2_VOLTAGE + R1_VOLTAGE) / (3537 * RL_CURRENT * R1_VOLTAGE * 1000);

    return power;
}

float getEnergy()
{
    uint32_t parm = 0;
    parm = ((uint32_t)SerialTemps[15] << 16) + ((uint32_t)SerialTemps[14] << 8) + SerialTemps[13];
    float energy = (float)parm * 1638.4 * 256 * V_REF * V_REF * (R2_VOLTAGE + R1_VOLTAGE) / (3600000.0 * 3537 * RL_CURRENT * R1_VOLTAGE * 1000);

    return energy;
}

float getFREQ()
{
    uint32_t parm = 0;
    parm = ((uint32_t)SerialTemps[18] << 16) + ((uint32_t)SerialTemps[17] << 8) + SerialTemps[16];
    float FREQ = 0;
    if (parm > 0)
    {
        FREQ = 500 * 1000 * 2 / (float)parm;
    }

    return FREQ;
}

int getSTATUS()
{
    return (int)SerialTemps[19];
}