/*
File:bl0937.h
Memo:读取计量芯片的电能数据，尝试兼容不同的计量芯片
Date:2023/6/2
Coder:oldfox126@foxmail.com
目的：为新国标五孔计量插座10A(https://oshwhub.com/oldfox126/xin-guo-biao-wu-kong-ji-liang-cha-zuo-10a) 配套的软件工程，本插座使用新国标五孔外壳，相比老款外壳更容易布线。在保留老款特性(磁保持继电器、电能计量、OTA升级固件)的同时，也解决了最后的短板(电流计数不稳定、外壳无法完美贴合等)。是计量插座的完全体。
开源协议: CC-BY-NC-SA 3.0
*/

#include "private/bl0937.h"

float current_multiplier; // Unit: us/A
float voltage_multiplier; // Unit: us/V
float power_multiplier;   // Unit: us/W
float current = 0;
float voltage = 0;
float power = 0;
float energy = 0;
int power_pulse_count = 0;
int voltage_pulse_count = 0;
int current_pulse_count = 0;
bool CF1IsVol = true;
int Pcount = 0;
int Vcount = 0;
int Ccount = 0;
unsigned long totalCF = 0;

void InitReadChip()
{
    pinMode(CF, INPUT_PULLUP);
    pinMode(CF1, INPUT_PULLUP);
    pinMode(SEL, OUTPUT);

    calculateDefaultMultipliers();
    attachInterrupt(CF1, CF1Interrupt, HIGH);
    attachInterrupt(CF, CFInterrupt, HIGH);

    readyReadVoltage();
}

void calculateDefaultMultipliers()
{
    // 1000000.0 *
    power_multiplier = (V_REF * V_REF * R_VOLTAGE / R_CURRENT) / 1721506;
    voltage_multiplier = (R_VOLTAGE * V_REF) / 15397;
    current_multiplier = (V_REF / R_CURRENT) / 94638;
}

String getDebugInfo()
{
    char buffer[300];
    sprintf(buffer, "KWh0=%.4f,KWh=%.4f,pulse_count[Voltage/Current/Power/TotalCF]=%d/%d/%d/%ld,[Voltage/Current/Power/Energy]=%.2f/%.2f/%.2f/%.4f,rate[Voltage/Current/Power]=%.3f/%.3f/%.3f", KWh0, sconfig.KWh, voltage_pulse_count, current_pulse_count, power_pulse_count, totalCF, voltage, current, power, energy, mconfig.vrate, mconfig.crate, mconfig.prate);
    return String(buffer);
}

IRAM_ATTR void CFInterrupt()
{
    power_pulse_count++;
    totalCF++;
}

IRAM_ATTR void CF1Interrupt()
{
    if (CF1IsVol)
    {
        voltage_pulse_count++;
        return;
    }

    current_pulse_count++;
}

// 每秒钟读一次，依次读(电压或者电流，和功率)。
char cnt = -1;
void readChip()
{
    readPower();

    if (CF1IsVol)
    {
        readVoltage();
        readyReadCurrent();
    }
    else
    {
        readCurrent();
        readyReadVoltage();
    }

    cnt++;
    float kwh = getEnergy();
    if (0 >= cnt) // first read
    {
        KWh0 = sconfig.KWh - kwh;
        if (0 > KWh0)
        {
            KWh0 = 0;
        }
        sconfig.KWh = KWh0 + kwh;
        needSaveStatus = 1;
        return;
    }

    sconfig.KWh = KWh0 + kwh;
    if (cnt >= 60)
    {
        cnt = 0;
        needSaveStatus = 1;
    }
}

void readPower()
{
    power_pulse_count = power_pulse_count - Pcount;
    Pcount = power_pulse_count;
    power = power_pulse_count * power_multiplier * mconfig.prate;        // W
    energy = totalCF * power_multiplier * mconfig.prate / (1000 * 3600); // KWH
}

float getActivePower()
{
    return power;
}

void readyReadVoltage()
{
    CF1IsVol = true;
    digitalWrite(SEL, HIGH);
}

void readVoltage()
{
    voltage_pulse_count = voltage_pulse_count - Vcount;
    Vcount = voltage_pulse_count;
    voltage = voltage_pulse_count * voltage_multiplier * mconfig.vrate; // V
}

float getVoltage()
{
    return voltage;
}

void readyReadCurrent()
{
    CF1IsVol = false;
    digitalWrite(SEL, LOW);
}

void readCurrent()
{
    current_pulse_count = current_pulse_count - Ccount;
    Ccount = current_pulse_count;
    current = current_pulse_count * current_multiplier * mconfig.crate * 1000; // mA
}

float getCurrent()
{
    // 功率大于0才有电流
    if (0 >= power)
    {
        current = 0;
    }

    return current;
}

float getEnergy()
{
    return energy;
}